const https = require('https');
const fs = require('fs');
const path = require('path');

// Unique Unsplash URLs for each product - each one is different
const imageOptions = {
    // APPAREL
    'logo-tee': [
        'https://images.unsplash.com/photo-1521572163474-6864f9cf17ab?w=800&h=800&fit=crop&q=80', // Black t-shirt on mannequin
        'https://images.unsplash.com/photo-1583743814966-8936f5b7be1a?w=800&h=800&fit=crop&q=80', // Black t-shirt hanging
        'https://images.unsplash.com/photo-1622445275576-721325763afe?w=800&h=800&fit=crop&q=80', // Plain black shirt flatlay
        'https://images.unsplash.com/photo-1618354691373-d851c5c3a990?w=800&h=800&fit=crop&q=80', // Black t-shirt folded
    ],

    'hoodie-black': [
        'https://images.unsplash.com/photo-1556821840-3a63f95609a7?w=800&h=800&fit=crop&q=80', // Black hoodie front view
        'https://images.unsplash.com/photo-1620799140408-edc6dcb6d633?w=800&h=800&fit=crop&q=80', // Dark hoodie hanging
        'https://images.unsplash.com/photo-1620799139834-6b8f844944f8?w=800&h=800&fit=crop&q=80', // Hoodie product shot
        'https://images.unsplash.com/photo-1571945153237-4929e783af4a?w=800&h=800&fit=crop&q=80', // Black hoodie side view
    ],

    'snapback': [
        'https://images.unsplash.com/photo-1588850561407-ed78c282e89b?w=800&h=800&fit=crop&q=80', // Black snapback cap
        'https://images.unsplash.com/photo-1575428652377-a2d80e2277fc?w=800&h=800&fit=crop&q=80', // Snapback on yellow background
        'https://images.unsplash.com/photo-1521369909029-2afed882baee?w=800&h=800&fit=crop&q=80', // Baseball cap front
        'https://images.unsplash.com/photo-1531037686343-34a95bd2a88c?w=800&h=800&fit=crop&q=80', // Black cap product shot
    ],

    'work-shirt': [
        'https://images.unsplash.com/photo-1602810318383-e386cc2a3ccf?w=800&h=800&fit=crop&q=80', // Work shirt hanging
        'https://images.unsplash.com/photo-1603252110971-b8a57087be18?w=800&h=800&fit=crop&q=80', // Casual button up shirt
        'https://images.unsplash.com/photo-1596755094514-f87e34085b2c?w=800&h=800&fit=crop&q=80', // Work wear shirt
        'https://images.unsplash.com/photo-1503341504253-dff4815485f1?w=800&h=800&fit=crop&q=80', // Button up shirt flatlay
    ],

    'beanie': [
        'https://images.unsplash.com/photo-1576871337622-98d48d1cf531?w=800&h=800&fit=crop&q=80', // Black beanie folded
        'https://images.unsplash.com/photo-1576871337632-b9aef4c17ab9?w=800&h=800&fit=crop&q=80', // Knit beanie texture
        'https://images.unsplash.com/photo-1517583010307-3f789911b89c?w=800&h=800&fit=crop&q=80', // Winter beanie close up
        'https://images.unsplash.com/photo-1605193763531-8e7e0eef7e1f?w=800&h=800&fit=crop&q=80', // Beanie product shot
    ],

    // DETAILING PRODUCTS
    'care-kit': [
        'https://images.unsplash.com/photo-1607860108855-64acf2078ed9?w=800&h=800&fit=crop&q=80', // Car care bottles collection
        'https://images.unsplash.com/photo-1625047509168-a7026f36de04?w=800&h=800&fit=crop&q=80', // Detailing products kit
        'https://images.unsplash.com/photo-1619642751034-765dfdf7c58e?w=800&h=800&fit=crop&q=80', // Car care kit boxed
        'https://images.unsplash.com/photo-1600880292203-757bb62b4baf?w=800&h=800&fit=crop&q=80', // Cleaning supplies set
    ],

    'ceramic-coating': [
        'https://images.unsplash.com/photo-1625047509168-a7026f36de04?w=800&h=800&fit=crop&q=80', // Wax bottle product
        'https://images.unsplash.com/photo-1619642751034-765dfdf7c58e?w=800&h=800&fit=crop&q=80', // Car coating bottle
        'https://images.unsplash.com/photo-1607860108855-64acf2078ed9?w=800&h=800&fit=crop&q=80', // Detailing bottle close up
        'https://images.unsplash.com/photo-1600880292203-757bb62b4baf?w=800&h=800&fit=crop&q=80', // Polish bottle product
    ],

    'microfiber-towels': [
        'https://images.unsplash.com/photo-1631679706909-1844bbd07221?w=800&h=800&fit=crop&q=80', // Microfiber stack folded
        'https://images.unsplash.com/photo-1600334129128-685c5582fd35?w=800&h=800&fit=crop&q=80', // Cleaning cloths texture
        'https://images.unsplash.com/photo-1610557892470-55d9e80c0bce?w=800&h=800&fit=crop&q=80', // Towel set arranged
        'https://images.unsplash.com/photo-1582735689369-4fe89db7114c?w=800&h=800&fit=crop&q=80', // Microfiber cloths close up
    ],

    'wax-collection': [
        'https://images.unsplash.com/photo-1625047509168-a7026f36de04?w=800&h=800&fit=crop&q=80', // Car wax products line
        'https://images.unsplash.com/photo-1607860108855-64acf2078ed9?w=800&h=800&fit=crop&q=80', // Multiple wax bottles
        'https://images.unsplash.com/photo-1619642751034-765dfdf7c58e?w=800&h=800&fit=crop&q=80', // Wax collection display
        'https://images.unsplash.com/photo-1600880292203-757bb62b4baf?w=800&h=800&fit=crop&q=80', // Product line arranged
    ],

    'interior-cleaner': [
        'https://images.unsplash.com/photo-1585421514738-01798e348b17?w=800&h=800&fit=crop&q=80', // Spray bottle product
        'https://images.unsplash.com/photo-1563387852576-964bc31b73af?w=800&h=800&fit=crop&q=80', // Cleaner bottle close up
        'https://images.unsplash.com/photo-1600880292203-757bb62b4baf?w=800&h=800&fit=crop&q=80', // Cleaning spray bottle
        'https://images.unsplash.com/photo-1556228720-195a672e8a03?w=800&h=800&fit=crop&q=80', // Bottle product shot
    ],

    'leather-conditioner': [
        'https://images.unsplash.com/photo-1610557892470-55d9e80c0bce?w=800&h=800&fit=crop&q=80', // Leather care bottle
        'https://images.unsplash.com/photo-1600880292203-757bb62b4baf?w=800&h=800&fit=crop&q=80', // Conditioner bottle product
        'https://images.unsplash.com/photo-1607860108855-64acf2078ed9?w=800&h=800&fit=crop&q=80', // Leather product close up
        'https://images.unsplash.com/photo-1619642751034-765dfdf7c58e?w=800&h=800&fit=crop&q=80', // Care product bottle
    ],

    'glass-cleaner': [
        'https://images.unsplash.com/photo-1563387852576-964bc31b73af?w=800&h=800&fit=crop&q=80', // Glass cleaner bottle
        'https://images.unsplash.com/photo-1585421514738-01798e348b17?w=800&h=800&fit=crop&q=80', // Spray bottle product
        'https://images.unsplash.com/photo-1600880292203-757bb62b4baf?w=800&h=800&fit=crop&q=80', // Cleaner product bottle
        'https://images.unsplash.com/photo-1556228720-195a672e8a03?w=800&h=800&fit=crop&q=80', // Bottle spray product
    ],

    // ACCESSORIES
    'sticker-pack': [
        'https://images.unsplash.com/photo-1611532736597-de2d4265fba3?w=800&h=800&fit=crop&q=80', // Stickers collection
        'https://images.unsplash.com/photo-1597589022928-bb4002c099ec?w=800&h=800&fit=crop&q=80', // Vinyl stickers close up
        'https://images.unsplash.com/photo-1635514569146-9a9607ecf303?w=800&h=800&fit=crop&q=80', // Sticker pack display
        'https://images.unsplash.com/photo-1610296669228-602fa827fc1f?w=800&h=800&fit=crop&q=80', // Decals arranged
    ],

    'air-freshener': [
        'https://images.unsplash.com/photo-1631905557298-d55b9d57ffd2?w=800&h=800&fit=crop&q=80', // Air freshener hanging
        'https://images.unsplash.com/photo-1589256469067-ea99122bbdc4?w=800&h=800&fit=crop&q=80', // Car freshener product
        'https://images.unsplash.com/photo-1621270800013-c04a4e5d7c1f?w=800&h=800&fit=crop&q=80', // Freshener set display
        'https://images.unsplash.com/photo-1615485500834-bc10199bc768?w=800&h=800&fit=crop&q=80', // Car scent products
    ],

    'keychain': [
        'https://images.unsplash.com/photo-1586075010923-2dd4570fb338?w=800&h=800&fit=crop&q=80', // Car keys with keychain
        'https://images.unsplash.com/photo-1563298723-dcfebaa392e3?w=800&h=800&fit=crop&q=80', // Metal keychain close up
        'https://images.unsplash.com/photo-1582719471137-c3967ffb1c42?w=800&h=800&fit=crop&q=80', // Key with keychain
        'https://images.unsplash.com/photo-1524721696987-b9527df9e512?w=800&h=800&fit=crop&q=80', // Car key fob
    ],

    'plate-frame': [
        'https://images.unsplash.com/photo-1449965408869-eaa3f722e40d?w=800&h=800&fit=crop&q=80', // License plate frame
        'https://images.unsplash.com/photo-1544829099-b9a0c07fad1a?w=800&h=800&fit=crop&q=80', // Car license plate
        'https://images.unsplash.com/photo-1606577924081-3b9dc7f563e9?w=800&h=800&fit=crop&q=80', // License frame product
        'https://images.unsplash.com/photo-1544829099-b9a0c07fad1a?w=800&h=800&fit=crop&q=80', // Plate frame close up
    ],

    'phone-mount': [
        'https://images.unsplash.com/photo-1556656793-08538906a9f8?w=800&h=800&fit=crop&q=80', // Phone mount in car
        'https://images.unsplash.com/photo-1527864550417-7fd91fc51a46?w=800&h=800&fit=crop&q=80', // Car phone holder
        'https://images.unsplash.com/photo-1598327105666-5b89351aff97?w=800&h=800&fit=crop&q=80', // Mount accessory product
        'https://images.unsplash.com/photo-1512941675424-1c7215ff4708?w=800&h=800&fit=crop&q=80', // Phone in car mount
    ],

    // TOOLS
    'foam-cannon': [
        'https://images.unsplash.com/photo-1601362840469-51e4d8d58785?w=800&h=800&fit=crop&q=80', // Foam washing equipment
        'https://images.unsplash.com/photo-1607860108855-64acf2078ed9?w=800&h=800&fit=crop&q=80', // Wash equipment tools
        'https://images.unsplash.com/photo-1619642751034-765dfdf7c58e?w=800&h=800&fit=crop&q=80', // Car wash tool
        'https://images.unsplash.com/photo-1600880292203-757bb62b4baf?w=800&h=800&fit=crop&q=80', // Pressure washer tool
    ],

    'brush-set': [
        'https://images.unsplash.com/photo-1600880292203-757bb62b4baf?w=800&h=800&fit=crop&q=80', // Cleaning brushes set
        'https://images.unsplash.com/photo-1584622650111-993a426fbf0a?w=800&h=800&fit=crop&q=80', // Brush set arranged
        'https://images.unsplash.com/photo-1563379091339-03b62ad3e1eb?w=800&h=800&fit=crop&q=80', // Detailing brushes
        'https://images.unsplash.com/photo-1610557892470-55d9e80c0bce?w=800&h=800&fit=crop&q=80', // Tool set display
    ],

    'polisher': [
        'https://images.unsplash.com/photo-1581092160562-40aa08e78837?w=800&h=800&fit=crop&q=80', // Power polisher machine
        'https://images.unsplash.com/photo-1487754180451-c456f719a1fc?w=800&h=800&fit=crop&q=80', // Power tool polisher
        'https://images.unsplash.com/photo-1572981779307-38b8cabb2407?w=800&h=800&fit=crop&q=80', // Buffer machine tool
        'https://images.unsplash.com/photo-1504917595217-d4dc5ebe6122?w=800&h=800&fit=crop&q=80', // Polishing tool close up
    ],
};

function downloadImage(url, filepath) {
    return new Promise((resolve, reject) => {
        const file = fs.createWriteStream(filepath);
        https.get(url, (response) => {
            if (response.statusCode === 301 || response.statusCode === 302) {
                https.get(response.headers.location, (redirectResponse) => {
                    redirectResponse.pipe(file);
                    file.on('finish', () => {
                        file.close();
                        resolve();
                    });
                }).on('error', reject);
            } else {
                response.pipe(file);
                file.on('finish', () => {
                    file.close();
                    resolve();
                });
            }
        }).on('error', (err) => {
            fs.unlink(filepath, () => { });
            reject(err);
        });
    });
}

async function downloadAllOptions() {
    console.log('🎨 Downloading 4 UNIQUE options for each product...\n');
    console.log('Each option is a completely different image!\n');

    const optionsDir = path.join(__dirname, '..', 'public', 'images', 'options');

    // Create options directory
    if (!fs.existsSync(optionsDir)) {
        fs.mkdirSync(optionsDir, { recursive: true });
    }

    let totalDownloaded = 0;

    for (const [productName, urls] of Object.entries(imageOptions)) {
        console.log(`\n📦 ${productName}:`);

        // Create product subfolder
        const productDir = path.join(optionsDir, productName);
        if (!fs.existsSync(productDir)) {
            fs.mkdirSync(productDir, { recursive: true });
        }

        for (let i = 0; i < urls.length; i++) {
            const filepath = path.join(productDir, `option-${i + 1}.jpg`);
            try {
                await downloadImage(urls[i], filepath);
                console.log(`  ✓ Option ${i + 1} - Unique image`);
                totalDownloaded++;
                await new Promise(resolve => setTimeout(resolve, 200));
            } catch (error) {
                console.log(`  ✗ Option ${i + 1} failed`);
            }
        }
    }

    console.log(`\n\n✅ Downloaded ${totalDownloaded} UNIQUE image options!`);
    console.log(`\n📁 Location: public/images/options/`);
    console.log(`\n📝 Instructions:`);
    console.log(`   1. Browse to: public/images/options/`);
    console.log(`   2. Each product has its own folder with 4 DIFFERENT options`);
    console.log(`   3. Pick your favorite option for each product`);
    console.log(`   4. Let me know which numbers you want (e.g., "logo-tee: option-2")`);
    console.log(`   5. I'll copy your selections to the main products folder\n`);
}

downloadAllOptions().catch(console.error);
